﻿// ----------------------------------------------------------------------------------------------------------------------------------
// Updates:
// ----------------------------------------------------------------------------------------------------------------------------------
//
// [October 28, 2009]	- Paul Dinelle - Update:  Altered some functions to allow more function calls 
//						  ------------------------------------------------------------------------
//							
//
// [February 27, 2009]	- Paul Dinelle - Update:  Fixed bugs to audio functions
//						  ------------------------------------------------------------------------
//							Fixed several bugs relating to audio functions and the timers, which were brought to my attention by
//							Kevin and Masoud.  This involved clearing issues in the template, which has been fitted with a new function
//							called "clearForNextSound".
//
//
// [December 19, 2008]  - Paul Dinelle - Update:  Recoded this class to have Pausing functionality
//						  ------------------------------------------------------------------------
//							Several changes were made to accomodate the new standards from the Ministry.  A "pause" functionality had
//							to be included, as well as a "muting" (the current mute actually shut off the sound, which is no longer
//							supposed to happen.  The following items were added:
//
//								Variables
//								---------
//									otherSound:Sound;
//									otherSoundChannel:SoundChannel;
//									pausedInterval_sndChannel:Number;
//									pausedInterval_otherSoundChannel:Number;
//									isPaused:Boolean;
//									timerPaused:Boolean;
//
//								Properties
//								---------
//									audioIsLoaded
//									paused
//
//								Functions
//								---------
//									stopAllAudio
//									pauseAudio
//									resumeAudio
//									clearOtherSound
//									allSoundsStopped
//									
//							
//
// [November 26, 2008]	- Paul Dinelle - Update:  Added in volume control
//						  -----------------------------------------------
//							You can now change the overall volume of the audio files.  Individual volume control is not currently
//							available, primarily due to an annoying lack of a feature in Flash.
//
//
// [July 29, 2008] 		- Paul Dinelle - Minor Update:  Added in different "onComplete" functionality
//						  ---------------------------------------------------------------------------
//					 		If the audio is muted (even in the middle of a playing sound file), the stored function is executed.
//					 		This change only applies to the "setSound" method of playing sounds.  "playSound" does not store
//							functions, nor does it execute them.
//
// ----------------------------------------------------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------------------------------------------------------------
// Things to note:
//		- "soundDone" function is stored even after it is executed.  This allows for the Replay function to still execute this
//				stored function after the replay.  If you need it to clear this stored function, simply go to the "performSoundDone"
//				function and follow the instructions.
// ----------------------------------------------------------------------------------------------------------------------------------


package {
	import flash.media.Sound;
	import flash.media.SoundChannel;
	import flash.media.SoundTransform;
	import flash.media.SoundMixer;
	import flash.utils.Timer;
	import flash.events.TimerEvent;
	import flash.events.Event;
	
	public class Audio {
		
		// Declare vars here.
		private var snd:Sound;
		private var sndChannel:SoundChannel;
		private var pausedInterval:Number;
		private var canReplay:Boolean;
		
		private var tPause:int;
		private var timer:Timer;
		private var isPaused:Boolean;
		private var timerPaused:Boolean;
		private var useDelay:Boolean;
		
		private var prevPrompt:Array;
		private var promptArray:Array;
		private var curPrompt:int;
		
		private var clearFuncAfter:Boolean;
		private var bMuted:Boolean;
		private var soundIsPlaying:Boolean;
		
		private var aBank:AudioBank;
		private var soundDone:Function;
		
		private var isLoaded:Boolean;
		private var theVolume:Number;
		
		
		
		// Constructor.
		public function Audio() {
			// No sound as yet.
			snd        = null;
			sndChannel = null;
			soundDone  = null;
			prevPrompt = null;
			promptArray = null;
			curPrompt = -1;
			clearFuncAfter = false;
			theVolume = SoundMixer.soundTransform.volume;
			pausedInterval = 0;
			timerPaused = false;
			canReplay = true;
			
			// No sound is playing.
			soundIsPlaying = false;
			
			// No audio bank either.
			aBank = null;
			
			// Not muted.
			bMuted = false;
			
			// AudioBank is not loaded yet.
			isLoaded = false;
			
			// Audio is not paused by default.
			isPaused = false;
			
			// Set a default pause time.
			tPause = 500;
			
			// Create a timer.
			timer = new Timer(tPause, 1);
		}
		
		
		// XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
		// XXXX      SETTERS AND GETTERS      XXXX
		// XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
		

		public function set delay(t:Number):void {
			// Purpose:  Sets the pause time (in seconds).
			
			// Set the delay.
			tPause = int(t * 1000);
			
			// Create a new timer.
			timer.delay = tPause;
		}
		public function set bank(bnk:AudioBank):void {
			// Purpose:  Sets the audio bank for this instance.
			aBank = bnk;
		}
		public function set onComplete(aFunc:Function):void {
			// Purpose:  Sets a function to be called when sound has finished playing.
			soundDone = aFunc;
		}
		public function set mute(bool:Boolean):void {
			// Purpose:  Sets the mute state.
			bMuted = bool;
			
			// Muting causes the sound volume to be set to zero, but does not change the "theVolume" variable.
			// Thus, once this is no longer mute, the volume is returned to its defined level.
			setMixerVolume();
		}
		public function get mute ():Boolean {
			return (bMuted);
		}
		public function set audioIsLoaded (aBool:Boolean):void {
			isLoaded = aBool;
			setMixerVolume();
		}
		public function set volume (aNum:Number):void {
			// Sets "theVolume" variable, and then updates the current volume of sounds.
			setVolume(aNum);
			setMixerVolume();
		}
		public function get volume ():Number {
			return (theVolume);
		}
		public function get paused ():Boolean {
			return (isPaused);
		}
		public function get isPlaying ():Boolean {
		// Purpose: Returns whether or not a sound is playing.
			return (soundIsPlaying);
		}
		public function get clearEndFunction ():Boolean {
			return( clearFuncAfter );
		}
		public function set clearEndFunction (aBool:Boolean):void {
			clearFuncAfter = aBool;
		}
		
		
		// XXXXXXXXXXXXXXXXXXXXXXXXXXXXX
		// XXXX      FUNCTIONS      XXXX
		// XXXXXXXXXXXXXXXXXXXXXXXXXXXXX
		
		
		public function clear():void {
			// Purpose:  Clears the prompt and sound.
			
			clearAllSoundInfo();
			clearMisc();
			clearPrompts();
			clearPausing();
		}
		public function clearForNextSound():void {
			clearCurSoundInfo();
			clearMisc();
			resetSequencePrompt();
			clearPausing();
		}
		public function clearPausing ():void {
			isPaused = false;
			pausedInterval = 0;
			timerPaused = false;
		}
		
		// These clear functions are for keeping the "clear" and "clearForNextSound" functions clean:
		private function clearMisc ():void {
			clearFuncAfter = false;
		}
		private function clearPrompts ():void {
			prevPrompt = null;
			promptArray = null;
			resetSequencePrompt();
		}
		private function resetSequencePrompt ():void {
			curPrompt = -1;
		}
		private function clearAllSoundInfo ():void {
			snd = null;
			soundDone = null;
			
			clearCurSoundInfo();
		}
		private function clearCurSoundInfo ():void {
			timer.reset();
			timer.removeEventListener(TimerEvent.TIMER, onTimedOut);
			
			if ( sndChannel != null ) {
				sndChannel.stop();
				sndChannel = null;
			}
			
			soundIsPlaying = false;
		}
		
		public function stopAllAudio ():void {
			// This function is used as a partial substitute for SoundMixer.stopAll()
			// The reason is that, when you progress to the next/previous screen, you need to clear
			// the audio so that it is not replayed when the user clicks Replay.
			
			SoundMixer.stopAll();
			clear();
		}
		public function setSound( somePrompts:Array, saveForReplay:Boolean = true, willUseDelay:Boolean = true ):void {
			// Purpose:  Sets the current prompt and plays it.
			
			if ( somePrompts == null  ||  (somePrompts is Array) == false ) {
				throw new Error ("[ERROR] (Audio.as) Function \"setSound\" was sent either a non-Array parameter, a null parameter, or an empty Array. Please resolve.");
				return;
			}
			
			promptArray = somePrompts;
			canReplay = saveForReplay;
			useDelay = willUseDelay;
			
			if ( canReplay ) {
				prevPrompt = promptArray;
			}
			
			// If audio isn't loaded,
			if (isLoaded == false) {
				// Don't play audio, but execute the function.
				performSoundDone();
				return;
			}
			
			performTheNextSound();
		}
		public function replay():void {
			// Purpose:  Replays the current prompt.
			
			// If the audio isn't loaded,
			if ( isLoaded == false  ||  isPaused ) {
				// Don't play audio.
				return;
			}
			
			// Play the last sound (or sound array) that was played.
			curPrompt = -1;
			
			// If the current sound cannot be replayed, use the previous one.
			if ( canReplay == false ) {
				if ( prevPrompt == null ) {
					// Don't replay the sound
					return;
				} else {
					promptArray = prevPrompt;
				}
			}
			
			playPrompt();
		}
		public function pauseAudio ():void {
			isPaused = true;
			
			if (timer.running) {
				timerPaused = true;
				timer.stop();
			}
			
			// If there is something being played on the sndChannel,
			if (sndChannel != null) {
				// Record this channel's playhead position:
				pausedInterval = sndChannel.position;
				
				// Stop this channel's sound:
				sndChannel.stop();
			}
		}
		public function resumeAudio ():void {
			isPaused = false;
			
			// If there was something being played on the sndChannel,
			if (sndChannel != null) {
				// Resume playing from the pause point:
				sndChannel = snd.play( pausedInterval );
				sndChannel.addEventListener(Event.SOUND_COMPLETE, sndComplete, false, 0, true);
				
				// Clear that pause point:
				pausedInterval = 0;
			}
			
			if (timerPaused) {
				timerPaused = false;
				// Call the function that the timer would have called when it was triggered.
				onTimedOut(null);
			}
		}
		
		private function onTimedOut(evt:TimerEvent):void {
			// Purpose:  Plays the prompt.
			
			// Remove the listener.
			timer.removeEventListener(TimerEvent.TIMER, onTimedOut);
			timer.reset();
			
			// If the prompt has not been cleared in the meantime,
			if ( promptArray != null ) {
				// Play the prompt.
				playPrompt();
			}
		}
		private function playPrompt():void {
			// Purpose: Plays the prompt.
			
			if ( promptArray != null ) {
				// Get the sound to play.
				curPrompt++;
				snd = aBank.getAudio( promptArray[curPrompt] );
			}
			
			// If the sound isn't in the bank,
			if (snd == null) {
				// Ignore it, and execute function.
				performSoundDone();
				return;
			}
			
			// Play it.
			sndChannel = snd.play();
			
			// Reset the interval that "Pausing" saves.
			pausedInterval = 0;
			// Sets the volume in case it hasn't been set yet.
			setMixerVolume();
			
			// Sound is now playing.
			soundIsPlaying = true;
			
			// Listen for the end.
			sndChannel.addEventListener(Event.SOUND_COMPLETE, sndComplete, false, 0, true);
		}
		private function sndComplete(evt:Event):void {
			// Purpose:  Performs the action that user wants done when the audio is completed.
			
			// Remove the listener.
			if ( sndChannel != null ) {
				sndChannel.removeEventListener(Event.SOUND_COMPLETE, sndComplete);
			}
			// Clear the channel:
			sndChannel = null;
			
			// Check that no sounds are playing.
			if ( allSoundsStopped() ) {
				soundIsPlaying = false;
			}
			
			// Perform the user's action.
			performSoundDone();
		}
		private function performSoundDone ():void {
			if ( isLoaded  &&  promptArray != null  &&  curPrompt < promptArray.length - 1 ) {
				performTheNextSound();
				return;
			}
			
			if ( clearFuncAfter ) {
				var tempFunction:Function = soundDone;
				soundDone = null;
				tempFunction();
			} else {
				if (soundDone != null) { soundDone(); }
			}
			
		}
		private function performTheNextSound ():void {
			// Sound has started:
			soundIsPlaying = true;
			
			if ( useDelay ) {
				// Set and start the timer.
				timer.addEventListener(TimerEvent.TIMER, onTimedOut, false, 0, true);
				timer.start();
			} else {
				playPrompt();
			}
		}
		private function endSoundEarly ():void {
		// Purpose:  Ends the sound early (if the mute option is set) and plays the function "soundDone" if it exists.
			soundIsPlaying = false;
			
			performSoundDone();
		}
		
		private function setVolume (aNum:Number = 1):void {
			// Sets the volume variable, but does not automatically update the actual sound's volume.
			
			// If aNum is specified:
			if (arguments.length > 0) {
				// Make sure it's not a negative number:
				if (aNum < 0) { aNum = 0; }
				// Make sure it's not greater than 1:
				else if (aNum > 1) { aNum = 1; }
			}
			
			// Set the volume variable:
			theVolume = aNum;
		}
		private function setMixerVolume ():void {
			// Sets the current volume to 0 if muted, or to whatever is specified in "theVolume"
			
			// Get the current soundTransform from the SoundMixer:
			var tempSndTransform:SoundTransform = SoundMixer.soundTransform;
			
			// If the audio is supposed to be muted,
			if (bMuted) {
				// Set the volume to zero:
				tempSndTransform.volume = 0;
			} else {
				// Otherwise set it to whatever "theVolume" is:
				tempSndTransform.volume = theVolume;
			}
			
			// Set the SoundMixer's transform to the modified transform:
			SoundMixer.soundTransform = tempSndTransform;
		}
		
		private function allSoundsStopped ():Boolean {
			// Checks to see if there is anything being played on either of the two channels
			
			if ( sndChannel != null ) {
				return (false);
			}
			
			return (true);
		}
		
	}  // End: class Audio
}  // End: package